/*
 * HD44780.c driver library for HD44780 LCD Displays
 *
 *  Created on: 08.04.2016
 *      Author: Nicolas Dammin
 */
#include "stm32f1xx_hal.h"
#include "HD44780.h"

volatile uint32_t time_ms;

void HAL_SYSTICK_Callback(void) {
	time_ms++;
}

void _delay_us(uint16_t wait) {
	TIM2->CNT = 0;
	while(TIM2->CNT < wait);
}

void _delay_ms(uint32_t wait) {
	uint32_t end = time_ms+wait;
	while(time_ms < end);
}

/* PRIVATE FUNCTIONS */
void writePin(uint32_t pin, uint8_t state) {
	if(pin == LCD_DB7_Pin || pin == LCD_RS_Pin){
		HAL_GPIO_WritePin(GPIOA, pin, state);
	} else {
		HAL_GPIO_WritePin(GPIOB, pin, state);
	}
}

void LCD_Enable(void) {
	writePin(LCD_E_Pin, GPIO_PIN_SET);
	_delay_us(20);
	writePin(LCD_E_Pin, GPIO_PIN_RESET);
	//_delay_us(20);
}

void LCD_Out(uint8_t data) {
	data &= 0xF0;

	if(data & 0x80) HAL_GPIO_WritePin(LCD_DB7_GPIO_Port, LCD_DB7_Pin, GPIO_PIN_SET); else HAL_GPIO_WritePin(LCD_DB7_GPIO_Port, LCD_DB7_Pin, GPIO_PIN_RESET);
	if(data & 0x40) HAL_GPIO_WritePin(LCD_DB6_GPIO_Port, LCD_DB6_Pin, GPIO_PIN_SET); else HAL_GPIO_WritePin(LCD_DB6_GPIO_Port, LCD_DB6_Pin, GPIO_PIN_RESET);
	if(data & 0x20) HAL_GPIO_WritePin(LCD_DB5_GPIO_Port, LCD_DB5_Pin, GPIO_PIN_SET); else HAL_GPIO_WritePin(LCD_DB5_GPIO_Port, LCD_DB5_Pin, GPIO_PIN_RESET);
	if(data & 0x10) HAL_GPIO_WritePin(LCD_DB4_GPIO_Port, LCD_DB4_Pin, GPIO_PIN_SET); else HAL_GPIO_WritePin(LCD_DB4_GPIO_Port, LCD_DB4_Pin, GPIO_PIN_RESET);

	LCD_Enable();
}

/* PUBLIC FUNCTIONS */
void LCD_Command(uint8_t data) {
	writePin(LCD_RS_Pin, GPIO_PIN_RESET);

	LCD_Out(data);
	LCD_Out(data<<4);

	_delay_us(42);
}

void LCD_Data(uint8_t data) {
	writePin(LCD_RS_Pin, GPIO_PIN_SET);

	LCD_Out(data);
	LCD_Out(data<<4);

	_delay_us(46);
}

void HD44780_Clear(void) {
	LCD_Command(LCD_CLEAR_DISPLAY);
	_delay_ms(2);
}

void HD44780_Home(void) {
	LCD_Command(LCD_CURSOR_HOME);
	_delay_ms(2);
}

void HD44780_Set_Cursor(uint8_t x, uint8_t y) {
	uint8_t data;

	switch (y)
	{
		case 1:    // 1st line
			data = LCD_SET_DDADR + LCD_DDADR_LINE1 + x;
			break;

		case 2:    // 2nd line
			data = LCD_SET_DDADR + LCD_DDADR_LINE2 + x;
			break;

		case 3:    // 3rd line
			data = LCD_SET_DDADR + LCD_DDADR_LINE3 + x;
			break;

		case 4:    // 4th line
			data = LCD_SET_DDADR + LCD_DDADR_LINE4 + x;
			break;

		default:
			return;                                   // if wrong line was selected
	}

	LCD_Command(data);
}

void HD44780_Init(uint8_t lines, uint8_t columns, uint8_t linemode)
{

    // wait for LCD to boot up
	_delay_ms(45);

	// Soft-Reset has to be sent 3 times in a row for initialization
	writePin(LCD_DB5_Pin, GPIO_PIN_SET);
	writePin(LCD_DB4_Pin, GPIO_PIN_SET);
	writePin(LCD_DB7_Pin, GPIO_PIN_RESET);
	writePin(LCD_DB6_Pin, GPIO_PIN_RESET);
	LCD_Enable();
    _delay_ms(5);

    LCD_Enable();
    _delay_ms(1);

    LCD_Enable();
    _delay_ms(1);

    // Enable 4-Bit Mode
    LCD_Out( LCD_SET_FUNCTION | LCD_FUNCTION_4BIT );
    _delay_ms(5);

    // 4-bit Mode / 2 Lines / 5x7
    LCD_Command(LCD_SET_FUNCTION |
                LCD_FUNCTION_4BIT |
                LCD_FUNCTION_2LINE |
                LCD_FUNCTION_5X8);

    // Display on / Cursor on / Blinking on
    LCD_Command(LCD_SET_DISPLAY |
                LCD_DISPLAY_ON |
                LCD_CURSOR_ON |
                LCD_BLINKING_ON);

    // Cursor increment / no Scrolling
    LCD_Command(LCD_SET_ENTRY |
                LCD_ENTRY_INCREASE |
                LCD_ENTRY_NOSHIFT);

    HD44780_Clear();
}

void HD44780_Char(char data) {
	LCD_Data(data);
}

void HD44780_String(const char *data)
{
    while(*data != '\0')
    	LCD_Data(*data++);
}

void LCD_GenerateChar(uint8_t code, const uint8_t *data)
{
    // Set startposition of the custom char
    LCD_Command( LCD_SET_CGADR | (code<<3) );

    // Transfer Bitmask
    for (uint8_t i=0; i<8; i++)
    {
        LCD_Data(data[i]);
    }
}




